//*************************************************************************************************
//
//	Description:
//		DepthOfField.fx
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Alastair Murray
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		AMurray		    01/10/2007  0.1           Created
//		TMann			05/11/2007	1.9			  Modified for PS3
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"


texture sceneTex : TEXTURE;
sampler sceneInputTex : SAMPLER = sampler_state
{
#ifdef _PS3_
	FX_SAMPLERSTATE_SRGB_TEXTURE
#else
	FX_SAMPLERSTATE_LINEAR_TEXTURE
#endif
	Texture = < sceneTex >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture blur5x5Tex : TEXTURE;
sampler blur5x5InputTex : SAMPLER = sampler_state
{
#ifdef _PS3_
	FX_SAMPLERSTATE_SRGB_TEXTURE
#else
	FX_SAMPLERSTATE_LINEAR_TEXTURE
#endif
	Texture = < blur5x5Tex >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture blurTex : TEXTURE;
sampler blurInputTex : SAMPLER = sampler_state
{
#ifdef _PS3_
	FX_SAMPLERSTATE_SRGB_TEXTURE
#else
	FX_SAMPLERSTATE_LINEAR_TEXTURE
#endif
	Texture = < blurTex >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture heatHazeNormalTex : TEXTURE;
sampler heatHazeNormalInputTex : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < heatHazeNormalTex >;
	AddressU  = Wrap;
	AddressV  = Wrap;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture depthTex : TEXTURE;
sampler depthInputTex : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < depthTex >;
	AddressU  = Clamp;
	AddressV  = Clamp;
#ifdef _PS3_
	MinFilter = Point;
	MagFilter = Point;
#else
	MinFilter = Linear;
	MagFilter = Linear;
#endif	
	MipFilter = None;
	SET_NO_ANISOTROPY
};

float2 viewportOrigin;
float2 viewportScale;
float2 DownsampleOffsets[ 9 ];


struct VSINPUT
{
	float3 position : POSITION;
	float2 texCoord : TEXCOORD0;
};

struct VSOUTPUT
{
	float4 position : POSITION;
	float2 texCoord : TEXCOORD0;
};


VSOUTPUT DepthOfFieldVS( VSINPUT _input )
{
	VSOUTPUT output;

	output.position = float4( _input.position.xyz, 1.0f );
	output.texCoord = _input.texCoord;

	return output;
}


VSOUTPUT DepthOfFieldVS_WithViewport( VSINPUT _input )
{
	VSOUTPUT output;

	// Apply the viewport transformation to the input tex coord
	output.position = float4( _input.position.xyz, 1.0f );
	output.texCoord = ( _input.texCoord * viewportScale ) + viewportOrigin;

	return output;
}

// DOF_SceneClipPlanes
// x is the near clip plane distance of the current scene camera
// y is [far/(far-near)] of the current scene camera
float4	DOF_SceneClipPlanes;

// DOF_FocusPlanes
//		Anything nearer than dist[0] is totally unfocused
//		Anything between dist[0] and dist[1] is interpolated from unfocused to focused
//		Anything between dist[1] and dist[2] is totally in focus
//		Anything between dist[2] and dist[3] is interpolated from focused to unfocused
//		Anything further than dist[3] is totally unfocused
float4	DOF_FocusPlanes;

// DOF_FocusPlanes_yMinusx_wMinusZ
// x = DOF_FocusPlanes.y - DOF_FocusPlanes.x
// y = DOF_FocusPlanes.w - DOF_FocusPlanes.z
float2	DOF_FocusPlanes_yMinusx_wMinusZ;

// DOF_MaxBlur (values are 0-1)
// x = MaxBlur for near side
// y = MaxBlur for far distance
float2	DOF_MaxBlur;

float	CalcDepth( float4 depthTex )
{
#ifdef _PS3_
	// calc depth using z buffer
	const float3 depthFactor = float3(65536.0f / 16777215.0f, 256.0f / 16777215.0f, 1.0f / 16777215.0f);
	float depth = dot(round(float3(depthTex.a, depthTex.r, depthTex.g) * 255.0f), depthFactor);	
#else
	// calc depth
	#ifdef _XBOX
		float	depth = depthTex.x;
	#else
		#ifdef _DEPTH_FROM_ZBUFFER_
			float	depth = (depthTex.x*255.0f/256.0f) + (depthTex.y*255.0f/65536.0f) + (depthTex.z*255.0f/16777216.0f);
		#else
			float	depth = depthTex.x;
		#endif
	#endif
#endif	

	return depth;
}


float	CalcSceneZ( float4 depthTex )
{
	float	depth = CalcDepth( depthTex );

	// depth to camera space
	float	sceneZ = (-DOF_SceneClipPlanes.x * DOF_SceneClipPlanes.y) / (depth-DOF_SceneClipPlanes.y);

	return sceneZ;
}


float	CalcBlurAmount( float4 depthTex )
{
	float	depth = CalcDepth( depthTex );
	float	blurAmount;

	if( depth==0.0f )
	{
		// sky is rendered with depth of zero rather than one.
		// this is because we want to limit how much blurring we do on the sky
		// in order to avoid it leaking in to the scenary
		blurAmount = 0.4f;
		blurAmount *= DOF_MaxBlur.y;
	}
	else
	{
		// depth to camera space
		float	sceneZ = (-DOF_SceneClipPlanes.x * DOF_SceneClipPlanes.y) / (depth-DOF_SceneClipPlanes.y);
	
		// calc blur amount based upon focal lengths
		if( sceneZ<DOF_FocusPlanes.y )	// we're on the near side of the focal point
		{
			blurAmount = sceneZ-DOF_FocusPlanes.x;					// so if we're less than 0, we want maximum blur
			blurAmount /= (DOF_FocusPlanes_yMinusx_wMinusZ.x);	// so if we're more than 1, we want no blur
			blurAmount = 1.0f - saturate( blurAmount );
			blurAmount *= DOF_MaxBlur.x;
		}
		else
		{
			blurAmount = sceneZ-DOF_FocusPlanes.z;					// so if we're less than 0, we want no blur
			blurAmount /= (DOF_FocusPlanes_yMinusx_wMinusZ.y);	// so if we're more than 1, we want maximum blur
			blurAmount = saturate( blurAmount );
			blurAmount *= DOF_MaxBlur.y;
		}
	
		//blurAmount = 1.0f-blurAmount;
		//blurAmount *= blurAmount;
		//blurAmount *= blurAmount;
		//blurAmount = 1.0f-blurAmount;
	}

	return blurAmount;
}

// this one outputs -1 for near blur, 0 for focus, +1 for far blur
float	CalcBlurAmountDepth( float4 depthTex )
{
	float	depth = CalcDepth( depthTex );
	float	blurAmount;

	if( depth==0.0f )
	{
		// sky is rendered with depth of zero rather than one.
		// this is because we want to limit how much blurring we do on the sky
		// in order to avoid it leaking in to the scenary
		blurAmount = 0.4f;
		blurAmount *= DOF_MaxBlur.y;
	}
	else
	{
		// depth to camera space
		float	sceneZ = (-DOF_SceneClipPlanes.x * DOF_SceneClipPlanes.y) / (depth-DOF_SceneClipPlanes.y);
	
		// calc blur amount based upon focal lengths
		if( sceneZ<DOF_FocusPlanes.y )	// we're on the near side of the focal point
		{
			blurAmount = sceneZ-DOF_FocusPlanes.x;							// so if we're less than 0, we want maximum blur
			blurAmount /= (DOF_FocusPlanes_yMinusx_wMinusZ.x);	// so if we're more than 1, we want no blur
			blurAmount = saturate( blurAmount ) - 1.0f;
			blurAmount *= DOF_MaxBlur.x;
		}
		else
		{
			blurAmount = sceneZ-DOF_FocusPlanes.z;					// so if we're less than 0, we want no blur
			blurAmount /= (DOF_FocusPlanes_yMinusx_wMinusZ.y);	// so if we're more than 1, we want maximum blur
			blurAmount = saturate( blurAmount );
			blurAmount *= DOF_MaxBlur.y;
		}
	
		//blurAmount = 1.0f-blurAmount;
		//blurAmount *= blurAmount;
		//blurAmount *= blurAmount;
		//blurAmount = 1.0f-blurAmount;
	}

	return blurAmount;
}


float2	MotionBlur_Planes;			// x=near, y=far   (before the near, everything is considered 'close' so is blurred, after the far, everything is considered 'distant' so is not blurred)
float	MotionBlur_MaxBlur;			// between 0 and 1

float	CalcMotionBlurAmount( float4 depthTex )
{
	float	depth = CalcDepth( depthTex );
	float	blurAmount;

	if( depth==0.0f )
	{
		// sky is rendered with depth of zero rather than one.
		blurAmount = 0.7f;	// allow motion blur across the sky even though it's very distant
	}
	else
	{
		// depth to camera space
		float	sceneZ = (-DOF_SceneClipPlanes.x * DOF_SceneClipPlanes.y) / (depth-DOF_SceneClipPlanes.y);
	
		// calc blur amount based upon distance of this pixel from the camera
		blurAmount = 1.0f - saturate( (sceneZ-MotionBlur_Planes.x) / (MotionBlur_Planes.y-MotionBlur_Planes.x) );
		//blurAmount *= blurAmount;
		//blurAmount *= blurAmount;
	}

	blurAmount *= MotionBlur_MaxBlur;

	return blurAmount;
}


float2	heatHazeDistances = float2(120.0f,600.0f);	// x = distance from camera until heat haze starts, y = distance from there until it's maximum intensity
float	heatHazeMaxIntensity = 12.0f;
float	heatHazeTile = 1.0f;
float	currentTime;

float2	CalcHeatHazeUVOffset( float2 uv, float sceneZ, out float3 normal, out float heathaze )
{
	heathaze = saturate( (sceneZ - heatHazeDistances.x)/heatHazeDistances.y ) * heatHazeMaxIntensity;
	float2	normalTexUV = uv*heatHazeTile;
	normalTexUV.y += currentTime*0.125f;
	normal = (tex2D( heatHazeNormalInputTex, normalTexUV ).rgb*2.0f)-1.0f;

	normalTexUV = uv*heatHazeTile + float2(0.15f,0.32f);
	normalTexUV.x += currentTime*0.045f;
	normalTexUV.y += currentTime*0.115f;
	normal += (tex2D( heatHazeNormalInputTex, normalTexUV ).rgb*2.0f)-1.0f;
	normal = normalize( normal );

	float2	offset = normal.xy;
	offset *= heathaze;
	offset.x *= 1.0f / 1920.0f;
	offset.y *= 1.0f / 1080.0f;
	uv += offset;

	return uv;
}

COLOUR_OUTPUT_TYPE DepthOfFieldPS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	output;
	
#ifdef _PS3_
//	float4	depthTex = tex2D( depthInputTex, uv );
//	float4	blurTex = tex2D( blurInputTex, uv );

//	float blurAmount = CalcBlurAmount( depthTex );

//	output.rgb=blurTex.rgb;
//	output.a=blurAmount;

	// get depth and calculate blurring for this pixel
	float4	depthTex = tex2D( depthInputTex, uv );
	float blurAmount = CalcBlurAmount( depthTex );

	// sample scene and blur texture
	float3	sceneTex = tex2D( sceneInputTex, uv );
	float4	blurTex = tex2D( blurInputTex, uv );

	// get output based on blurAmount
	output.rgb = lerp( sceneTex.rgb, blurTex.rgb, blurAmount );

	//output.rgb = blurAmount;

	output.a = 1.0f;
#else	

	// get depth and calculate blurring for this pixel
	float4	depthTex = tex2D( depthInputTex, uv );
	float blurAmount = CalcBlurAmount( depthTex );

	// sample scene and blur texture
	float3	sceneTex = tex2D( sceneInputTex, uv );
	float4	blurTex = tex2D( blurInputTex, uv );

	// get output based on blurAmount
	output.rgb = lerp( sceneTex.rgb, blurTex.rgb, blurAmount );
	
	//output.rgb = blurAmount;
	//output.rgb = 1.0f-((1.0f-blurAmount)*(1.0f-blurAmount));
	//output.rgb = 1.0f-((1.0f-blurAmount)*(1.0f-blurAmount)*(1.0f-blurAmount));

	output.a = 1.0f;
#endif	

	return output;
}

COLOUR_OUTPUT_TYPE	DepthOfFieldWithHeatHazePS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	output;
	
	// get depth and calculate blurring for this pixel
	float4	depthTex = tex2D( depthInputTex, uv );
	float blurAmount = CalcBlurAmount( depthTex );

	// calc heat haze offset
	float	sceneZ = CalcSceneZ( depthTex );
	float3	normal;
	float	heathaze;
	uv = CalcHeatHazeUVOffset( uv, sceneZ, normal, heathaze );

	// sample scene and blur texture
	float3	sceneTex = tex2D( sceneInputTex, uv );
	float4	blurTex = tex2D( blurInputTex, uv );

	// get output based on blurAmount
	output.rgb = lerp( sceneTex.rgb, blurTex.rgb, blurAmount );

	//output.rgb = blurAmount;
	//output.rgb = lerp( output.rgb, (normal+1.0f)*0.5f, heathaze/heatHazeMaxIntensity );

	output.a = 1.0f;

	return output;
}


COLOUR_OUTPUT_TYPE	MotionBlurPS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	output;
	
#ifdef _PS3_
	float4	depthTex = tex2D( depthInputTex, uv );
	float4	blurTex = tex2D( blurInputTex, uv );

	float blurAmount = CalcMotionBlurAmount( depthTex );

	output.rgb=blurTex.rgb;
	output.a=blurAmount;
#else	

	// get textures
	float4	depthTex = tex2D( depthInputTex, uv );
	float3	sceneTex = tex2D( sceneInputTex, uv );
	float4	blurTex = tex2D( blurInputTex, uv );

	float blurAmount = CalcMotionBlurAmount( depthTex );

	// get output based on blurAmount
	output.rgb = lerp( sceneTex.rgb, blurTex.rgb, blurAmount );
	//output.rgb = blurAmount;
	output.a = 1.0f;
#endif	

	return output;
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Downsample Scene whilst placing Blur amount in the Alpha

COLOUR_OUTPUT_TYPE	DownsamplePS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	output = tex2D( sceneInputTex, uv );
	output.a = CalcBlurAmount( tex2D( depthInputTex, uv ) ); 
	return output;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Downsample Scene preserving Alpha

COLOUR_OUTPUT_TYPE	Downsample2PS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	output = tex2D( sceneInputTex, uv );
	return output;
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Downsample Scene whilst placing Blur amount in the Alpha

COLOUR_OUTPUT_TYPE	DownsampleForMotionBlurPS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	output = tex2D( sceneInputTex, uv );
	output.a = CalcMotionBlurAmount( tex2D( depthInputTex, uv ) ); 
	return output;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////




//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// 9x Gaussian Blur - Needs separate Horiz + Vertical passes
// The weighting for this blur comes from the depth of field focus input, so this blur is not suitable for any other use

float4	TapOffsVS[3];		// pixel offsets for 1,2,3 pixels from center
float4	TapOffsNegVS[3];	// the negative of the above (or can be another pixel if required - as for the motion blur)

float4	TapOffsPS[3];		// pixel offsets for 4,5,6 pixels from center
float4	TapOffsNegPS[3];	// the negative of the above (or can be another pixel if required - as for the motion blur)

struct	VS_BLUR_OUTPUT
{
	float4	position : POSITION;
	float2	tap0 : TEXCOORD0;
	float2	tap1 : TEXCOORD1;
	float2	tap2 : TEXCOORD2;
	float2	tap3 : TEXCOORD3;
	float2	tap1neg : TEXCOORD4;
	float2	tap2neg : TEXCOORD5;
	float2	tap3neg : TEXCOORD6;
};

VS_BLUR_OUTPUT	GaussianBlur_VS( VSINPUT _input )
{
	VS_BLUR_OUTPUT _output;

	_output.position = float4( _input.position.xyz, 1.0f );
	_output.tap0 = _input.texCoord;
	_output.tap1 = _input.texCoord+TapOffsVS[0].xy;
	_output.tap2 = _input.texCoord+TapOffsVS[1].xy;
	_output.tap3 = _input.texCoord+TapOffsVS[2].xy;
	_output.tap1neg = _input.texCoord+TapOffsNegVS[0].xy;
	_output.tap2neg = _input.texCoord+TapOffsNegVS[1].xy;
	_output.tap3neg = _input.texCoord+TapOffsNegVS[2].xy;

	return _output;
}

COLOUR_OUTPUT_TYPE GaussianBlurX_PS( VS_BLUR_OUTPUT _input ) : COLOR0
{
	float4 Color[7];
	float	 Blur[7];
	float4 ColorSum = 0.0f;	

	const float4 vThresh0 = float4(0.1f,0.3f,0.5f,-0.01f)*0.25f;
	const float4 vThresh1 = float4(0.6f,0.7f,0.8f,0.9f)*0.25f;
	
	float4	vWeights4;
	float3	vWeights3;

	// sample inner taps
	Color[6]  = tex2D( blurInputTex, _input.tap3neg); 
	Color[5]  = tex2D( blurInputTex, _input.tap2neg);
	Color[4]  = tex2D( blurInputTex, _input.tap1neg);
	Color[0]  = tex2D( blurInputTex, _input.tap0); 
	Color[1]  = tex2D( blurInputTex, _input.tap1);
	Color[2]  = tex2D( blurInputTex, _input.tap2); 
	Color[3]  = tex2D( blurInputTex, _input.tap3);
	Blur[0] = Color[0].a;
	Blur[1] = Color[1].a;
	Blur[2] = Color[2].a;
	Blur[3] = Color[3].a;
	Blur[4] = Color[4].a;
	Blur[5] = Color[5].a;
	Blur[6] = Color[6].a;

	// calc weights
	vWeights4.x = saturate( Blur[1] - vThresh0.x );
	vWeights4.y = saturate( Blur[2] - vThresh0.y );
	vWeights4.z = saturate( Blur[3] - vThresh0.z );
	vWeights4.w = saturate( Blur[0] - vThresh0.w );

	// sum weighted samples	
	ColorSum = Color[0] * vWeights4.w;
	ColorSum += Color[1] * vWeights4.x;	
	ColorSum += Color[2] * vWeights4.y;
	ColorSum += Color[3] * vWeights4.z;

	// sum weights
	float fWeightSum = dot( vWeights4, 1.0f );
	
	// calc remaining weights
	vWeights3.x = saturate( Blur[4] - vThresh0.x );
	vWeights3.y = saturate( Blur[5] - vThresh0.y );
	vWeights3.z = saturate( Blur[6] - vThresh0.z );

	// sum weighted samples	
	ColorSum += Color[4] * vWeights3.x;
	ColorSum += Color[5] * vWeights3.y;	
	ColorSum += Color[6] * vWeights3.z;

	// sum weights
	fWeightSum += dot( vWeights3, 1.0f );

	// compute texture coordinates for other taps
	float2 Tap4 = _input.tap0 + TapOffsPS[0].xy;
	float2 Tap5 = _input.tap0 + TapOffsPS[1].xy;
	float2 Tap6 = _input.tap0 + TapOffsPS[2].xy;
	float2 Tap4Neg = _input.tap0 + TapOffsNegPS[0].xy;
	float2 Tap5Neg = _input.tap0 + TapOffsNegPS[1].xy;
	float2 Tap6Neg = _input.tap0 + TapOffsNegPS[2].xy;
	
	// sample outer taps
	Color[5]  = tex2D( blurInputTex, Tap6Neg);
	Color[4]  = tex2D( blurInputTex, Tap5Neg); 
	Color[3]  = tex2D( blurInputTex, Tap4Neg);
	Color[0]  = tex2D( blurInputTex, Tap4); 
	Color[1]  = tex2D( blurInputTex, Tap5); 
	Color[2]  = tex2D( blurInputTex, Tap6); 
	Blur[0] = Color[0].a;
	Blur[1] = Color[1].a;
	Blur[2] = Color[2].a;
	Blur[3] = Color[3].a;
	Blur[4] = Color[4].a;
	Blur[5] = Color[5].a;
	
	// calc weights
	vWeights3.x = saturate( Blur[0] - vThresh1.x );
	vWeights3.y = saturate( Blur[1] - vThresh1.y );
	vWeights3.z = saturate( Blur[2] - vThresh1.z );

	// sum weighted samples	
	ColorSum += Color[0] * vWeights3.x;
	ColorSum += Color[1] * vWeights3.y;	
	ColorSum += Color[2] * vWeights3.z;

	// sum weights
	fWeightSum += dot( vWeights3, 1.0f );

	// calc remaining weights
	vWeights3.x = saturate( Blur[3] - vThresh1.x );
	vWeights3.y = saturate( Blur[4] - vThresh1.y );
	vWeights3.z = saturate( Blur[5] - vThresh1.z );

	// sum weighted samples	
	ColorSum += Color[3] * vWeights3.x;
	ColorSum += Color[4] * vWeights3.y;	
	ColorSum += Color[5] * vWeights3.z;

	// sum weights
	fWeightSum += dot( vWeights3, 1.0f );

	// average colour
	ColorSum.rgb /= fWeightSum;

	// put weight in alpha to be read back by next filter pass
	ColorSum.a = fWeightSum;

	return ColorSum;				
}

// in this pass, sceneInputTex is the result of GaussianBlurX_PS
COLOUR_OUTPUT_TYPE GaussianBlurY_PS( VS_BLUR_OUTPUT _input ) : COLOR0
{
	float4 Color[7];
	float4 ColorSum = 0.0f;	

	const float4 vWeights0 = float4(0.080f,0.075f,0.070f,0.100f);
	const float4 vWeights1 = float4(0.065f,0.060f,0.055f,0.050f);

	// sample inner taps
	Color[0]  = tex2D( blurInputTex, _input.tap0); 
	Color[1]  = tex2D( blurInputTex, _input.tap1);
	Color[2]  = tex2D( blurInputTex, _input.tap2); 
	Color[3]  = tex2D( blurInputTex, _input.tap3);
	Color[4]  = tex2D( blurInputTex, _input.tap1neg);
	Color[5]  = tex2D( blurInputTex, _input.tap2neg);
	Color[6]  = tex2D( blurInputTex, _input.tap3neg); 

	// mod colours by weights that are in alpha channel from last pass
	Color[0].rgb *= Color[0].a;
	Color[1].rgb *= Color[1].a;
	Color[2].rgb *= Color[2].a;
	Color[3].rgb *= Color[3].a;
	Color[4].rgb *= Color[4].a;
	Color[5].rgb *= Color[5].a;
	Color[6].rgb *= Color[6].a;

	// sum weighted samples	
	ColorSum = Color[0] * vWeights0.w;
	ColorSum += (Color[1] + Color[4]) * vWeights0.x;	
	ColorSum += (Color[2] + Color[5]) * vWeights0.y;
	ColorSum += (Color[3] + Color[6]) * vWeights0.z;

	// compute texture coordinates for other taps
	float2 Tap4 = _input.tap0 + TapOffsPS[0].xy;
	float2 Tap5 = _input.tap0 + TapOffsPS[1].xy;
	float2 Tap6 = _input.tap0 + TapOffsPS[2].xy;
	float2 Tap4Neg = _input.tap0 + TapOffsNegPS[0].xy;
	float2 Tap5Neg = _input.tap0 + TapOffsNegPS[1].xy;
	float2 Tap6Neg = _input.tap0 + TapOffsNegPS[2].xy;
	
	// sample outer taps
	Color[0]  = tex2D( blurInputTex, Tap4); 
	Color[1]  = tex2D( blurInputTex, Tap5); 
	Color[2]  = tex2D( blurInputTex, Tap6); 
	Color[3]  = tex2D( blurInputTex, Tap4Neg);
	Color[4]  = tex2D( blurInputTex, Tap5Neg); 
	Color[5]  = tex2D( blurInputTex, Tap6Neg);
	
	// mod colours by weights that are in alpha channel from last pass
	Color[0].rgb *= Color[0].a;
	Color[1].rgb *= Color[1].a;
	Color[2].rgb *= Color[2].a;
	Color[3].rgb *= Color[3].a;
	Color[4].rgb *= Color[4].a;
	Color[5].rgb *= Color[5].a;

	// sum weighted samples	
	ColorSum += (Color[0] + Color[3]) * vWeights1.x;	
	ColorSum += (Color[1] + Color[4]) * vWeights1.y;
	ColorSum += (Color[2] + Color[5]) * vWeights1.z;

	// average colour
	ColorSum.rgb /= ColorSum.a;

	return ColorSum;				
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Simple 5x5 Gaussian Blur

float4   SampleOffsets[13];
float4   SampleWeights[13];

float4 GaussBlur5x5PS( in float2 vScreenPosition : TEXCOORD0 ) : COLOR
{
    float4 vColor = 0.0f;

    for( int i=0; i < 13; i++ )
    {
      vColor += SampleWeights[i] * tex2D( blur5x5InputTex, vScreenPosition + SampleOffsets[i].xy );
    }

    return vColor;
}


// this version of the blur, will not blur from area's that are less blurry than the current sample (blurriness comes from the alpha channel)
float4 GaussBlurForDOF5x5PS( in float2 vScreenPosition : TEXCOORD0 ) : COLOR
{
		float		t = SampleWeights[0].x;
		float4	vColor = tex2D( blur5x5InputTex, vScreenPosition+SampleOffsets[0].xy ) * SampleWeights[0].x;
		float		centerBlur = vColor.a;
    for( int i=1; i < 13; i++ )
    {
			float4	tap = tex2D( blur5x5InputTex, vScreenPosition + SampleOffsets[i].xy );
			float		w = (tap.a>=centerBlur)? SampleWeights[i].x : 0.0f;
			vColor += w * tap;
			t += w;
    }
		
		vColor /= t;
	
    return vColor;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////



//#define	ALPHA_STORES_DEPTH	// if defined, the alpha value stores a bluriness from -1 (near dist out of focus) to 0 (focus) to +1 (far dist out of focus)

COLOUR_OUTPUT_TYPE CopyBlurToAlphaPS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	sceneTex = 0.0f;
	
	#ifdef ALPHA_STORES_DEPTH
	sceneTex.a = (CalcBlurAmountDepth( tex2D( depthInputTex, uv ) )+1.0f)*0.5f;
	#else
	sceneTex.a = CalcBlurAmount( tex2D( depthInputTex, uv ) );	// blurriness
	#endif
	
	return sceneTex;
}

// very simple DOF routine that interpolates between the full res, and blurred textures based upon the blur param stored in the alpha channel
COLOUR_OUTPUT_TYPE DepthOfField2PS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	sceneTex = tex2D( sceneInputTex, uv );
	float4	blurTex = tex2D( blurInputTex, uv);
	float4	output = lerp( sceneTex, blurTex, blurTex.a );
	return output;	
}	



//
// currently unused routine for DOF (kept 'just in case')
//
#define	NUM_TAPS	8
float2	pixelSizeLow,pixelSizeHigh;
float2	poisson[NUM_TAPS];
/*
poisson[0].Set( -0.419418f, -0.616039f, 0.0f, 0.0f );
poisson[1].Set( 0.440453f, -0.639399f, 0.0f, 0.0f );
poisson[2].Set( -0.757088f, 0.349334f, 0.0f, 0.0f );
poisson[3].Set( 0.574619f, 0.685879f, 0.0f, 0.0f );
poisson[4].Set( 0.527837f, -0.085868f, 0.0f, 0.0f );
poisson[5].Set( -0.040088f, 0.536087f, 0.0f, 0.0f );
poisson[6].Set( -0.670445f ,-0.179949f, 0.0f, 0.0f );
poisson[7].Set( 0.0f, 0.0f, 0.0f, 0.0f );
*/
COLOUR_OUTPUT_TYPE DepthOfField3PS( float2 uv : TEXCOORD0 ) : COLOR0
{
	float4	output;

	// sample at center point
	float4	sceneTex = tex2D( sceneInputTex, uv );
	//sceneTex.a = CalcBlurAmount( tex2D( depthInputTex, uv ) );	// blurriness
			
	// calc blur radius
	#ifdef ALPHA_STORES_DEPTH
	float discRadius = abs(sceneTex.a*2.0f-1.0f) * 5.0f;	// radius in pixels
	#else
	float discRadius = sceneTex.a * 5.0f;	// radius in pixels
	#endif
	float	discRadiusLow = discRadius*0.4f;	// radius to use on blurred texture
	
	output.rgb = sceneTex.rgb;
	output.a = 1.0f;	// weighting of center point is 1
	
	#ifdef _XBOX
	[unroll(8)] 
	#endif
	for( int i=0; i<NUM_TAPS; i++ )
	{
		// get coords to sample from
		float2	uvLow = uv + (pixelSizeLow * poisson[i] * discRadiusLow);
		float2	uvHigh = uv + (pixelSizeHigh * poisson[i] * discRadius);
		
		// sample textures
		float4	sampleLow = tex2D( blurInputTex, uvLow );
		float4	sampleHigh = tex2D( sceneInputTex, uvHigh );
		//sampleHigh.a = CalcBlurAmount( tex2D( depthInputTex, uvHigh ) );	// blurriness
		#ifdef ALPHA_STORES_DEPTH
		float4	tap = lerp( sampleHigh, sampleLow, abs(sampleHigh.a*2.0f-1.0f) );
		#else
		float4	tap = lerp( sampleHigh, sampleLow, sampleHigh.a );	
		#endif
			
		// leak reduction
		#ifdef ALPHA_STORES_DEPTH
		tap.a = (tap.a>=sceneTex.a)? 1.0f : abs(tap.a*2.0f-1.0f);	// reduce any leaks
		#else
		//tap.a = (tap.a>=sceneTex.a)? 1.0f : tap.a;	// reduce any leaks
		tap.a = saturate(tap.a/sceneTex.a);	// reduce leaks
		#endif
		//tap.a = 1.0f;
		
		output.rgb += tap.rgb * tap.a;
		output.a += tap.a;
	}
	output /= output.a;
	
	//output = tex2D( blurInputTex, uv);
	//output.rgb = sceneTex.a;
	//output.rgb = 1.0f-((1.0f-sceneTex.a)*(1.0f-sceneTex.a));
	//output.rgb = 1.0f-((1.0f-sceneTex.a)*(1.0f-sceneTex.a)*(1.0f-sceneTex.a));

	return output;
}




technique DepthOfField
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
//		AlphaBlendEnable = true;
//		BlendFunc=int2(SrcAlpha, OneMinusSrcAlpha);
//		BlendEquation=int(FuncAdd);

		VertexShader = compile sce_vp_rsx DepthOfFieldVS_WithViewport();
		PixelShader = compile sce_fp_rsx DepthOfFieldPS();
#else		
		AlphaBlendEnable = false;
		VertexShader = compile vs_3_0 DepthOfFieldVS_WithViewport();
		PixelShader = compile ps_3_0 DepthOfFieldPS();
#endif
	}
}

technique CopyBlurToAlpha
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx DepthOfFieldVS_WithViewport();
		PixelShader = compile sce_fp_rsx CopyBlurToAlphaPS();
#else		
		AlphaBlendEnable = false;
		VertexShader = compile vs_3_0 DepthOfFieldVS_WithViewport();
		PixelShader = compile ps_3_0 CopyBlurToAlphaPS();
#endif
	}
}

technique DepthOfField2
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx DepthOfFieldVS_WithViewport();
		PixelShader = compile sce_fp_rsx DepthOfField2PS();
#else		
		AlphaBlendEnable = false;
		VertexShader = compile vs_3_0 DepthOfFieldVS_WithViewport();
		PixelShader = compile ps_3_0 DepthOfField2PS();
#endif
	}
}

technique DepthOfFieldWithHeatHaze
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		AlphaBlendEnable = true;
		BlendFunc=int2(SrcAlpha, OneMinusSrcAlpha);
		BlendEquation=int(FuncAdd);

		VertexShader = compile sce_vp_rsx DepthOfFieldVS_WithViewport();
		PixelShader = compile sce_fp_rsx DepthOfFieldWithHeatHazePS();
#else		
		AlphaBlendEnable = false;
		VertexShader = compile vs_3_0 DepthOfFieldVS_WithViewport();
		PixelShader = compile ps_3_0 DepthOfFieldWithHeatHazePS();
#endif
	}
}


technique MotionBlur
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		AlphaBlendEnable = true;
		BlendFunc=int2(SrcAlpha, OneMinusSrcAlpha);
		BlendEquation=int(FuncAdd);

		VertexShader = compile sce_vp_rsx DepthOfFieldVS_WithViewport();
		PixelShader = compile sce_fp_rsx MotionBlurPS();
#else		
		VertexShader = compile vs_3_0 DepthOfFieldVS_WithViewport();
		PixelShader = compile ps_3_0 MotionBlurPS();
#endif
	}
}



technique GaussianBlurX
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx GaussianBlur_VS();
		PixelShader = compile sce_fp_rsx GaussianBlurX_PS();
#else		
		VertexShader = compile vs_3_0 GaussianBlur_VS();
		PixelShader = compile ps_3_0 GaussianBlurX_PS();
#endif
	}
}

technique GaussianBlurY
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx GaussianBlur_VS();
		PixelShader = compile sce_fp_rsx GaussianBlurY_PS();
#else		
		VertexShader = compile vs_3_0 GaussianBlur_VS();
		PixelShader = compile ps_3_0 GaussianBlurY_PS();
#endif
	}
}


technique DownsampleForDOF
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable = false;
		VertexShader = compile sce_vp_rsx DepthOfFieldVS();
		PixelShader = compile sce_fp_rsx DownsamplePS();
#else		
		VertexShader = compile vs_3_0 DepthOfFieldVS();
		PixelShader = compile ps_3_0 DownsamplePS();
#endif
	}
}


technique DownsampleForDOF2
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable = false;
		VertexShader = compile sce_vp_rsx DepthOfFieldVS();
		PixelShader = compile sce_fp_rsx Downsample2PS();
#else		
		VertexShader = compile vs_3_0 DepthOfFieldVS();
		PixelShader = compile ps_3_0 Downsample2PS();
#endif
	}
}


technique DownsampleForMotionBlur
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable = false;
		VertexShader = compile sce_vp_rsx DepthOfFieldVS();
		PixelShader = compile sce_fp_rsx DownsampleForMotionBlurPS();
#else		
		VertexShader = compile vs_3_0 DepthOfFieldVS();
		PixelShader = compile ps_3_0 DownsampleForMotionBlurPS();
#endif
	}
}


technique GaussianBlur5x5
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx DepthOfFieldVS();
		PixelShader = compile sce_fp_rsx GaussBlur5x5PS();
#else		
		VertexShader = compile vs_3_0 DepthOfFieldVS();
		PixelShader = compile ps_3_0 GaussBlur5x5PS();
#endif
	}
}

technique GaussianBlurForDOF5x5
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx DepthOfFieldVS();
		PixelShader = compile sce_fp_rsx GaussBlurForDOF5x5PS();
#else		
		VertexShader = compile vs_3_0 DepthOfFieldVS();
		PixelShader = compile ps_3_0 GaussBlurForDOF5x5PS();
#endif
	}
}
